//
//  ARCoreUIViewController.h
//  ARCore
//
//  Created by  jefby on 2017/6/26.
//  Copyright © 2017年 人工智能实验室. All rights reserved.
//

#import <UIKit/UIKit.h>
#import "AGARCoreDelegate.h"

#define ENVIRONMENT_PRODUCT (1)
#define ENVIRONMENT_SANDBOX (2)

/**
 接口类
 */
@interface AGARCoreUIViewController : UIViewController <UIAlertViewDelegate>

#pragma mark - Instance Methods

/**
 reset整个场景，会清除掉当前的config、targetId缓存
 */
- (void)rescanClick;


/**
 截取当前正在显示的内容
 
 @return UIImage对象
 */
- (UIImage*)snapshot;


/**
 是否需要隐藏掉当前默认的Focus Region UI, 默认显示
 
 @warning 当customFocusRegionUI设置为NO,调用才有效
 @param hidden hide or not
 */
- (void)setDefaultFocusRegionUIHidden:(BOOL)hidden;

/**
 是否需要隐藏掉当前默认的Prompting Label,
 包括相机和marker距离提示、使用文案提示等, 默认显示
 
 @warning 当customFocusRegionUI设置为NO,调用才有效
 @param hidden hide or not
 */
- (void)setDefaultPromptingLabelHidden:(BOOL)hidden;


/**
 是否需要隐藏掉默认的network变化提示信息

 @param hidden hide or not
 */
- (void)setDefaultNetworkTipsHidden:(BOOL)hidden;

/**
 开始追踪
 */
- (void)startObjectTracking;

/**
 停止追踪
 */
- (void)stopObjectTracking;

/**
 Destroy all resource includes camera, glview,
 must invoke manually
 */
- (void)destroyAll;


/// 云识别需要, 在open.taobao.com上申请的应用appkey
@property (nonatomic, copy) NSString *appKey;
/// 在open.taobao.com上申请的应用secret
@property (nonatomic, copy) NSString *secret;

#pragma mark - Optional

/// 可选
@property (nonatomic, copy) NSString *tel;

/**
 当前App运行的环境，默认是生产环境（外网可访问）
 如果需要修改的话，修改当前环境的值且修改对应的appkey和secret
 可选的值为1，2.
 其中1表示生产环境，2表示测试环境（仅用于阿里巴巴内部开发者测试，无法访问外网资源）
 */
@property (nonatomic, assign) int currentEnvironment;

/**
 发向云识别服务端的UrlStr，因为不同环境(生产/测试)请求的URL不一样,
 默认为@"https://eco.taobao.com/router/rest"
 */ 
@property (nonatomic, copy) NSString *reqUrlStr;

/**
 当识别成功时需要播放的音效Url，目前仅支持本地音频
 */
@property (nonatomic, strong) NSURL *localAudioUrl;

/**
 是否需要定制扫描框的UI，默认是FALSE
 
 目前是有个默认的扫描框，扫描框的左上角坐标为(startX, startY)，宽高分别为
 scanWidth和scanHeight，计算方式如下：
 
 float screenWidth = self.view.frame.size.width;
 float screenHeight = self.view.frame.size.height;
 float scanWidth = (screenWidth + screenHeight) / 3.0f;
 float scanHeight = (screenWidth + screenHeight) / 2.5f;
 /// 处理裁剪的宽高大于屏幕宽高的情况，iPhone X上scanWidth > screenWidth
 if (scanWidth > screenWidth) {
    scanWidth = screenWidth - 40;
 }
 if (scanHeight > screenHeight) {
    scanHeight = screenHeight - 80;
 }
 startX = (screenWidth - scanWidth) / 2;
 startY = (screenHeight - scanHeight) / 2;
 */
@property (nonatomic, assign) BOOL customFocusRegionUI;

/**
 如果要定制扫描框的UI，需要设置Focus Region区域的左上角坐标和宽、高值
 */
/// Focus Region左上角x坐标(屏幕坐标)
@property (nonatomic, assign) float startX;
/// Focus Region左上角y坐标(屏幕坐标)
@property (nonatomic, assign) float startY;
/// Focus Region扫描框宽度(屏幕坐标)
@property (nonatomic, assign) float scanWidth;
/// Focus Region扫描框高度(屏幕坐标)
@property (nonatomic, assign) float scanHeight;

/**
 react rootview，顶层界面，给React Native使用，使用native开发的用户不需要关注
 */
@property (nonatomic) UIView* reactRootView;

/**
 用于判断是否可以通过pushViewcontroller或者presentViewcontroller显示该viewcontroller
 */
@property (nonatomic, readonly) BOOL canPresent;

/**
 AGARCoreDelegate代理，可选
 */
@property (nonatomic, weak) id<AGARCoreDelegate> delegate;


/**
 Top method api version , default is '2.3'
 */
@property (nonatomic, copy) NSString * topMethodVersion;


/**
 是否禁止3D Object Tracking
 */
@property (nonatomic) BOOL disable3DObjectTracking;


/**
 开启二维码识别功能
 */
@property (nonatomic) BOOL enableDetectQRCode;

/**
 创建Bubble effect
 */
- (void)createBubble;


/**
 更新Bubble文案
 
 @param image 合成后的图片
 */
- (void)updateBubbleTextWithImage:(UIImage*)image;


/**
 获取最多可创建的Bubble text数量
 
 @return 获取最大值
 */
- (unsigned int)getBubbleTextMaxNum;


/**
 销毁bubble effect
 */
- (void)destroyBubble;


@end
